(() => {
  "use strict";

  const STORAGE_KEYS = {
    users: "hims_users_v1",
    session: "hims_session_v1",
    appointments: "hims_appointments_v1",
    notifications: "hims_notifications_v1",
    reduceMotion: "hims_reduce_motion_v1"
  };

  const DEPARTMENTS = [
    "General Medicine",
    "Cardiology",
    "Neurology",
    "Orthopedics",
    "Pediatrics",
    "Dermatology"
  ];

  const dom = {};
  const state = {
    users: [],
    appointments: [],
    notifications: [],
    session: null,
    activePanel: "workspacePanel",
    activeStreamController: null
  };

  function uid(prefix) {
    return `${prefix}_${Math.random().toString(36).slice(2, 10)}`;
  }

  function readStore(key, fallback) {
    try {
      const raw = localStorage.getItem(key);
      return raw ? JSON.parse(raw) : fallback;
    } catch {
      return fallback;
    }
  }

  function writeStore(key, value) {
    localStorage.setItem(key, JSON.stringify(value));
  }

  function seedLocalData() {
    const users = readStore(STORAGE_KEYS.users, null);
    if (!users || !users.length) {
      writeStore(STORAGE_KEYS.users, [
        {
          id: "u_patient_seed",
          name: "Riya Sharma",
          email: "patient@hims.local",
          password: "123456",
          role: "patient",
          department: ""
        },
        {
          id: "u_doctor_seed",
          name: "Dr. Aisha Rahman",
          email: "doctor@hims.local",
          password: "123456",
          role: "doctor",
          department: "General Medicine"
        },
        {
          id: "u_admin_seed",
          name: "Admin Ops",
          email: "admin@hims.local",
          password: "123456",
          role: "admin",
          department: ""
        }
      ]);
    }

    const appointments = readStore(STORAGE_KEYS.appointments, null);
    if (!appointments || !appointments.length) {
      writeStore(STORAGE_KEYS.appointments, [
        {
          id: "appt_seed_1",
          patientId: "u_patient_seed",
          patientName: "Riya Sharma",
          doctorId: "u_doctor_seed",
          doctorName: "Dr. Aisha Rahman",
          department: "General Medicine",
          slot: toDateTimeLocalString(new Date(Date.now() + 3600 * 1000 * 8)),
          symptoms: "Headache for 2 days, mild nausea",
          reason: "General check",
          notes: "",
          status: "pending",
          aiSummary: "",
          createdAt: new Date().toISOString(),
          completedAt: null
        }
      ]);
    }

    const notifications = readStore(STORAGE_KEYS.notifications, null);
    if (!notifications) {
      writeStore(STORAGE_KEYS.notifications, []);
    }
  }

  function loadState() {
    state.users = readStore(STORAGE_KEYS.users, []);
    state.appointments = readStore(STORAGE_KEYS.appointments, []);
    state.notifications = readStore(STORAGE_KEYS.notifications, []);
    state.session = readStore(STORAGE_KEYS.session, null);
  }

  function persistAppointments() {
    writeStore(STORAGE_KEYS.appointments, state.appointments);
  }

  function persistNotifications() {
    writeStore(STORAGE_KEYS.notifications, state.notifications);
  }

  function addNotification(userId, message) {
    state.notifications.unshift({
      id: uid("notif"),
      userId,
      message,
      createdAt: new Date().toISOString()
    });
    persistNotifications();
  }

  function getDoctors() {
    return state.users.filter((user) => user.role === "doctor");
  }

  function getCurrentUser() {
    if (!state.session) return null;
    return state.users.find((user) => user.id === state.session.id) || null;
  }

  function switchToApp() {
    dom.authView.classList.add("hidden");
    dom.appView.classList.remove("hidden");
    dom.logoutBtn.classList.remove("hidden");
    renderUserMeta();
    renderWorkspace();
    renderAnalytics();
    switchPanel(state.activePanel);
  }

  function switchToAuth() {
    dom.authView.classList.remove("hidden");
    dom.appView.classList.add("hidden");
    dom.logoutBtn.classList.add("hidden");
    clearChat();
  }

  function applyAuthMode(mode) {
    const loginMode = mode === "login";
    dom.loginTab.classList.toggle("active", loginMode);
    dom.registerTab.classList.toggle("active", !loginMode);
    dom.loginForm.classList.toggle("hidden", !loginMode);
    dom.registerForm.classList.toggle("hidden", loginMode);
  }

  function toDateTimeLocalString(dateObj) {
    const date = new Date(dateObj);
    const year = date.getFullYear();
    const month = String(date.getMonth() + 1).padStart(2, "0");
    const day = String(date.getDate()).padStart(2, "0");
    const hours = String(date.getHours()).padStart(2, "0");
    const minutes = String(date.getMinutes()).padStart(2, "0");
    return `${year}-${month}-${day}T${hours}:${minutes}`;
  }

  function niceDate(value) {
    if (!value) return "N/A";
    const date = new Date(value);
    return new Intl.DateTimeFormat(undefined, {
      dateStyle: "medium",
      timeStyle: "short"
    }).format(date);
  }

  function metricCard(title, value) {
    return `
      <article class="metric">
        <p class="title">${escapeHtml(title)}</p>
        <p class="value">${escapeHtml(String(value))}</p>
      </article>
    `;
  }

  function escapeHtml(text) {
    return String(text)
      .replaceAll("&", "&amp;")
      .replaceAll("<", "&lt;")
      .replaceAll(">", "&gt;")
      .replaceAll('"', "&quot;")
      .replaceAll("'", "&#039;");
  }

  function showToast(message, type = "info") {
    if (!dom.toastWrap) {
      const wrap = document.createElement("div");
      wrap.className = "toast-wrap";
      document.body.appendChild(wrap);
      dom.toastWrap = wrap;
    }

    const toast = document.createElement("div");
    toast.className = `toast ${type}`;
    toast.textContent = message;
    dom.toastWrap.appendChild(toast);
    setTimeout(() => toast.remove(), 3200);
  }

  function renderUserMeta() {
    const user = getCurrentUser();
    if (!user) return;
    dom.userName.textContent = user.name;
    dom.userRole.textContent = `${user.role.toUpperCase()}${user.department ? ` • ${user.department}` : ""}`;
    dom.userEmail.textContent = user.email;
  }

  function renderWorkspace() {
    const user = getCurrentUser();
    if (!user) return;

    if (user.role === "patient") renderPatientWorkspace(user);
    if (user.role === "doctor") renderDoctorWorkspace(user);
    if (user.role === "admin") renderAdminWorkspace(user);
  }

  function renderPatientWorkspace(user) {
    const doctors = getDoctors();
    const doctorOptions = doctors.length
      ? doctors.map((doc) => `<option value="${doc.id}">${escapeHtml(doc.name)} (${escapeHtml(doc.department || "General")})</option>`).join("")
      : `<option value="">No doctors available</option>`;

    dom.workspacePanel.innerHTML = `
      <div class="panel-head">
        <h3>Patient Workspace</h3>
        <span class="chip">Book + Track + Follow-up</span>
      </div>
      <div class="workspace-grid">
        <section class="section-card">
          <h4>Book Appointment</h4>
          <p>Live slot validation prevents double booking for each doctor and time slot.</p>
          <form id="bookingForm" class="booking-form">
            <select name="department" required>
              ${DEPARTMENTS.map((dept) => `<option value="${dept}">${dept}</option>`).join("")}
            </select>
            <select name="doctorId" required>${doctorOptions}</select>
            <input type="datetime-local" name="slot" required min="${toDateTimeLocalString(new Date())}">
            <input type="text" name="reason" placeholder="Short reason (fever, headache, routine check)" required maxlength="120">
            <textarea name="symptoms" placeholder="Write symptoms for doctor + AI context..." required></textarea>
            <button type="submit" class="solid-btn">Confirm Booking</button>
          </form>
        </section>
        <section class="section-card">
          <h4>My Appointments</h4>
          <div id="patientAppointmentList" class="appointment-list"></div>
        </section>
        <section class="section-card">
          <h4>Follow-up Notifications</h4>
          <div id="patientNotificationList" class="appointment-list"></div>
        </section>
      </div>
    `;

    const bookingForm = document.getElementById("bookingForm");
    bookingForm.addEventListener("submit", (event) => {
      event.preventDefault();
      const formData = new FormData(bookingForm);
      const doctorId = formData.get("doctorId");
      const slot = formData.get("slot");
      const department = formData.get("department");
      const reason = String(formData.get("reason") || "").trim();
      const symptoms = String(formData.get("symptoms") || "").trim();

      const doctor = state.users.find((entry) => entry.id === doctorId && entry.role === "doctor");
      if (!doctor) {
        showToast("Selected doctor is not valid.", "error");
        return;
      }

      const hasConflict = state.appointments.some((appt) =>
        appt.doctorId === doctorId &&
        appt.slot === slot &&
        appt.status !== "cancelled"
      );
      if (hasConflict) {
        showToast("This slot is already booked. Pick a different time.", "error");
        return;
      }

      state.appointments.unshift({
        id: uid("appt"),
        patientId: user.id,
        patientName: user.name,
        doctorId,
        doctorName: doctor.name,
        department: department || doctor.department || "General Medicine",
        slot,
        reason,
        symptoms,
        notes: "",
        status: "pending",
        aiSummary: "",
        createdAt: new Date().toISOString(),
        completedAt: null
      });

      persistAppointments();
      addNotification(user.id, `Appointment request sent to ${doctor.name} for ${niceDate(slot)}.`);
      showToast("Appointment booked successfully.", "success");
      renderWorkspace();
      renderAnalytics();
    });

    const appointments = state.appointments
      .filter((appt) => appt.patientId === user.id)
      .sort((a, b) => new Date(b.createdAt) - new Date(a.createdAt));

    renderAppointmentList(document.getElementById("patientAppointmentList"), appointments, "patient");
    renderNotificationList(document.getElementById("patientNotificationList"), user.id);
  }

  function renderDoctorWorkspace(user) {
    dom.workspacePanel.innerHTML = `
      <div class="panel-head">
        <h3>Doctor Workspace</h3>
        <span class="chip">Queue + Completion Automation</span>
      </div>
      <div class="workspace-grid">
        <section class="section-card">
          <h4>Today's Queue</h4>
          <div id="doctorAppointmentList" class="appointment-list"></div>
        </section>
        <section class="section-card">
          <h4>Automated Summaries</h4>
          <div id="doctorSummaryList" class="appointment-list"></div>
        </section>
      </div>
    `;

    const doctorAppointments = state.appointments
      .filter((appt) => appt.doctorId === user.id)
      .sort((a, b) => new Date(a.slot) - new Date(b.slot));

    const completedWithSummary = doctorAppointments
      .filter((appt) => appt.status === "completed" && appt.aiSummary)
      .slice(0, 8);

    const listNode = document.getElementById("doctorAppointmentList");
    renderAppointmentList(listNode, doctorAppointments, "doctor");

    listNode.addEventListener("click", async (event) => {
      const button = event.target.closest("button[data-action]");
      if (!button) return;

      const action = button.dataset.action;
      const appointmentId = button.dataset.id;
      const target = state.appointments.find((appt) => appt.id === appointmentId && appt.doctorId === user.id);
      if (!target) {
        showToast("Appointment not found.", "error");
        return;
      }

      if (action === "approve") {
        target.status = "approved";
        persistAppointments();
        addNotification(target.patientId, `Your appointment with ${target.doctorName} is approved.`);
        showToast("Appointment approved.", "success");
        renderWorkspace();
        renderAnalytics();
      }

      if (action === "cancel") {
        target.status = "cancelled";
        persistAppointments();
        addNotification(target.patientId, `Your appointment with ${target.doctorName} was cancelled.`);
        showToast("Appointment cancelled.", "info");
        renderWorkspace();
        renderAnalytics();
      }

      if (action === "complete") {
        const consultationNotes = window.prompt(
          "Add consultation notes for AI summary generation:",
          target.notes || "Patient reviewed, treatment provided, monitor recovery for 5 days."
        );
        if (consultationNotes === null) return;

        target.notes = consultationNotes.trim();
        target.status = "completed";
        target.completedAt = new Date().toISOString();
        persistAppointments();
        renderWorkspace();
        renderAnalytics();
        showToast("Marked completed. Running follow-up AI agent...", "info");
        await runFollowUpAgent(target, user);
      }
    });

    const summaryNode = document.getElementById("doctorSummaryList");
    if (!completedWithSummary.length) {
      summaryNode.innerHTML = `<div class="empty-state">Complete an appointment to auto-generate discharge and follow-up summary.</div>`;
    } else {
      summaryNode.innerHTML = completedWithSummary.map((appt) => `
        <article class="appointment-card">
          <div class="card-head">
            <h4 class="appt-title">${escapeHtml(appt.patientName)} • ${escapeHtml(appt.department)}</h4>
            <span class="badge completed">completed</span>
          </div>
          <p class="appt-meta">${escapeHtml(niceDate(appt.slot))}</p>
          <p class="appt-notes">${escapeHtml(appt.aiSummary.slice(0, 420))}${appt.aiSummary.length > 420 ? "..." : ""}</p>
        </article>
      `).join("");
    }
  }

  function renderAdminWorkspace() {
    const totalUsers = state.users.length;
    const patientCount = state.users.filter((entry) => entry.role === "patient").length;
    const doctorCount = state.users.filter((entry) => entry.role === "doctor").length;
    const totalAppointments = state.appointments.length;
    const completionCount = state.appointments.filter((entry) => entry.status === "completed").length;

    dom.workspacePanel.innerHTML = `
      <div class="panel-head">
        <h3>Admin Workspace</h3>
        <span class="chip">Live oversight</span>
      </div>
      <div class="workspace-grid">
        <section class="section-card">
          <h4>System Snapshot</h4>
          <div class="metric-grid">
            ${metricCard("Total Users", totalUsers)}
            ${metricCard("Patients", patientCount)}
            ${metricCard("Doctors", doctorCount)}
            ${metricCard("Appointments", totalAppointments)}
          </div>
          <div class="metric-grid">
            ${metricCard("Completed Cases", completionCount)}
            ${metricCard("Completion Rate", totalAppointments ? `${Math.round((completionCount / totalAppointments) * 100)}%` : "0%")}
            ${metricCard("Automation Status", "Active")}
            ${metricCard("Data Layer", "LocalStorage")}
          </div>
        </section>
        <section class="section-card">
          <h4>All Appointments</h4>
          <div id="adminAppointmentList" class="appointment-list"></div>
        </section>
      </div>
    `;

    const appointments = [...state.appointments].sort((a, b) => new Date(b.createdAt) - new Date(a.createdAt));
    renderAppointmentList(document.getElementById("adminAppointmentList"), appointments, "admin");
  }

  function renderAppointmentList(container, appointments, viewerRole) {
    if (!appointments.length) {
      container.innerHTML = `<div class="empty-state">No records yet.</div>`;
      return;
    }

    const template = document.getElementById("appointmentCardTemplate");
    const fragment = document.createDocumentFragment();

    appointments.forEach((appt) => {
      const clone = template.content.cloneNode(true);
      clone.querySelector(".appt-title").textContent = `${appt.patientName} with ${appt.doctorName}`;
      clone.querySelector(".appt-status").textContent = appt.status;
      clone.querySelector(".appt-status").classList.add(statusClass(appt.status));
      clone.querySelector(".appt-meta").textContent = `${niceDate(appt.slot)} • ${appt.department}`;
      clone.querySelector(".appt-notes").textContent = `Symptoms: ${appt.symptoms || "N/A"}${appt.notes ? ` | Notes: ${appt.notes}` : ""}${appt.aiSummary ? ` | AI: ${appt.aiSummary.slice(0, 120)}${appt.aiSummary.length > 120 ? "..." : ""}` : ""}`;

      const actionWrap = clone.querySelector(".appt-actions");
      actionWrap.innerHTML = actionButtons(appt, viewerRole);
      fragment.appendChild(clone);
    });

    container.innerHTML = "";
    container.appendChild(fragment);
  }

  function actionButtons(appt, viewerRole) {
    if (viewerRole !== "doctor") return "";
    if (appt.status === "completed" || appt.status === "cancelled") {
      return `<button type="button" class="mini-btn refresh" data-action="noop" data-id="${appt.id}">Locked</button>`;
    }

    const buttons = [];
    if (appt.status === "pending") {
      buttons.push(`<button type="button" class="mini-btn approve" data-action="approve" data-id="${appt.id}">Approve</button>`);
    }
    buttons.push(`<button type="button" class="mini-btn complete" data-action="complete" data-id="${appt.id}">Complete + AI</button>`);
    buttons.push(`<button type="button" class="mini-btn cancel" data-action="cancel" data-id="${appt.id}">Cancel</button>`);
    return buttons.join("");
  }

  function statusClass(status) {
    if (status === "approved") return "approved";
    if (status === "completed") return "completed";
    if (status === "cancelled") return "cancelled";
    return "pending";
  }

  function renderNotificationList(container, userId) {
    const items = state.notifications.filter((entry) => entry.userId === userId).slice(0, 8);
    if (!items.length) {
      container.innerHTML = `<div class="empty-state">No follow-up notifications yet.</div>`;
      return;
    }

    container.innerHTML = items.map((item) => `
      <article class="appointment-card">
        <div class="card-head">
          <h4 class="appt-title">Care Update</h4>
          <span class="badge approved">new</span>
        </div>
        <p class="appt-notes">${escapeHtml(item.message)}</p>
        <p class="appt-meta">${escapeHtml(niceDate(item.createdAt))}</p>
      </article>
    `).join("");
  }

  async function runFollowUpAgent(appointment, doctor) {
    const prompt = [
      "Prepare a discharge and follow-up communication.",
      `Patient: ${appointment.patientName}`,
      `Doctor: ${doctor.name}`,
      `Department: ${appointment.department}`,
      `Symptoms: ${appointment.symptoms}`,
      `Consultation notes: ${appointment.notes || "No notes provided"}`,
      "Output format: plain text headings with colons, no markdown symbols, concise and warm."
    ].join("\n");

    let text = "";
    try {
      await streamAiResponse(
        {
          message: prompt,
          context: {
            mode: "follow_up_automation",
            role: "doctor"
          }
        },
        {
          onWord: (word) => {
            text += word;
          }
        }
      );

      const target = state.appointments.find((entry) => entry.id === appointment.id);
      if (target) {
        target.aiSummary = text.trim();
        persistAppointments();
      }

      addNotification(
        appointment.patientId,
        `Follow-up plan ready for your ${appointment.department} visit. Open appointment details to review care guidance.`
      );
      showToast("AI follow-up summary generated.", "success");
      loadState();
      renderWorkspace();
      renderAnalytics();
    } catch (error) {
      showToast(`AI follow-up failed: ${error.message}`, "error");
    }
  }

  function renderAnalytics() {
    const total = state.appointments.length;
    const pending = state.appointments.filter((entry) => entry.status === "pending").length;
    const approved = state.appointments.filter((entry) => entry.status === "approved").length;
    const completed = state.appointments.filter((entry) => entry.status === "completed").length;
    const cancelled = state.appointments.filter((entry) => entry.status === "cancelled").length;
    const completionRate = total ? `${Math.round((completed / total) * 100)}%` : "0%";

    const byDepartment = {};
    const byHour = {};
    state.appointments.forEach((entry) => {
      byDepartment[entry.department] = (byDepartment[entry.department] || 0) + 1;
      const hour = new Date(entry.slot).getHours();
      byHour[hour] = (byHour[hour] || 0) + 1;
    });

    const topDepartmentEntries = Object.entries(byDepartment).sort((a, b) => b[1] - a[1]).slice(0, 6);
    const peakHourEntry = Object.entries(byHour).sort((a, b) => b[1] - a[1])[0];
    const peakHour = peakHourEntry ? `${String(peakHourEntry[0]).padStart(2, "0")}:00` : "N/A";
    const maxDeptCount = topDepartmentEntries[0]?.[1] || 1;

    dom.analyticsPanel.innerHTML = `
      <div class="panel-head">
        <h3>Operational Insights</h3>
        <span class="chip">Computed locally</span>
      </div>
      <div class="metric-grid">
        ${metricCard("Total Appointments", total)}
        ${metricCard("Pending", pending)}
        ${metricCard("Approved", approved)}
        ${metricCard("Completed", completed)}
      </div>
      <div class="metric-grid">
        ${metricCard("Cancelled", cancelled)}
        ${metricCard("Completion Rate", completionRate)}
        ${metricCard("Peak Booking Hour", peakHour)}
        ${metricCard("Doctors Online", state.users.filter((u) => u.role === "doctor").length)}
      </div>
      <section class="section-card" style="margin-top:12px;">
        <h4>Department Demand</h4>
        <div class="bar-grid">
          ${topDepartmentEntries.length
            ? topDepartmentEntries.map(([department, count]) => `
              <div class="bar-row">
                <span>${escapeHtml(department)}</span>
                <div class="bar-track"><div class="bar-fill" style="width:${Math.max(6, Math.round((count / maxDeptCount) * 100))}%"></div></div>
                <strong>${count}</strong>
              </div>
            `).join("")
            : `<div class="empty-state">No department activity yet.</div>`
          }
        </div>
      </section>
    `;
  }

  function switchPanel(panelId) {
    state.activePanel = panelId;
    [dom.workspacePanel, dom.assistantPanel, dom.analyticsPanel].forEach((panel) => {
      panel.classList.add("hidden");
    });
    document.getElementById(panelId).classList.remove("hidden");
    dom.navButtons.forEach((btn) => {
      btn.classList.toggle("active", btn.dataset.panel === panelId);
    });
  }

  function appendMessage(role, text) {
    const div = document.createElement("div");
    div.className = `msg ${role === "user" ? "user" : "ai"}`;
    div.textContent = text;
    dom.chatStream.appendChild(div);
    dom.chatStream.scrollTop = dom.chatStream.scrollHeight;
    return div;
  }

  function clearChat() {
    dom.chatStream.innerHTML = "";
  }

  async function streamAiResponse(payload, callbacks = {}) {
    if (state.activeStreamController) {
      state.activeStreamController.abort();
    }
    state.activeStreamController = new AbortController();

    const response = await fetch("api/chat.php", {
      method: "POST",
      headers: {
        "Content-Type": "application/json"
      },
      body: JSON.stringify(payload),
      signal: state.activeStreamController.signal
    });

    if (!response.ok) {
      let message = `Request failed with ${response.status}`;
      try {
        const data = await response.json();
        if (data && data.error) message = data.error;
      } catch {
        // Keep default message.
      }
      throw new Error(message);
    }

    if (!response.body) {
      throw new Error("Streaming not supported in this browser.");
    }

    const reader = response.body.getReader();
    const decoder = new TextDecoder();
    let buffer = "";

    while (true) {
      const { done, value } = await reader.read();
      if (done) break;

      buffer += decoder.decode(value, { stream: true });
      let boundary;
      while ((boundary = buffer.indexOf("\n\n")) >= 0) {
        const rawEvent = buffer.slice(0, boundary).trim();
        buffer = buffer.slice(boundary + 2);
        if (!rawEvent) continue;

        const dataLines = rawEvent.split("\n").filter((line) => line.startsWith("data:"));
        for (const line of dataLines) {
          const payloadString = line.slice(5).trim();
          if (!payloadString) continue;

          let packet;
          try {
            packet = JSON.parse(payloadString);
          } catch {
            continue;
          }

          if (packet.error) {
            throw new Error(packet.error);
          }

          if (typeof packet.word === "string") {
            callbacks.onWord?.(packet.word);
          }

          if (packet.done) {
            callbacks.onDone?.(packet);
          }
        }
      }
    }
  }

  async function handleChatSubmit(event) {
    event.preventDefault();

    const message = dom.chatInput.value.trim();
    if (!message) return;

    const user = getCurrentUser();
    if (!user) {
      showToast("Session missing. Please login again.", "error");
      return;
    }

    dom.chatInput.value = "";
    appendMessage("user", message);
    const aiNode = appendMessage("ai", "");
    const typing = document.createElement("span");
    typing.className = "typing-dot";
    aiNode.appendChild(typing);
    dom.chatSubmit.disabled = true;

    let merged = "";
    try {
      await streamAiResponse(
        {
          message,
          context: {
            role: user.role,
            name: user.name
          }
        },
        {
          onWord: (word) => {
            merged += word;
            aiNode.textContent = merged;
            dom.chatStream.scrollTop = dom.chatStream.scrollHeight;
          }
        }
      );
    } catch (error) {
      aiNode.textContent = `Error: ${error.message}`;
      showToast(`AI request failed: ${error.message}`, "error");
    } finally {
      dom.chatSubmit.disabled = false;
      dom.chatStream.scrollTop = dom.chatStream.scrollHeight;
    }
  }

  function attachEvents() {
    dom.loginTab.addEventListener("click", () => applyAuthMode("login"));
    dom.registerTab.addEventListener("click", () => applyAuthMode("register"));

    dom.loginForm.addEventListener("submit", (event) => {
      event.preventDefault();
      const email = String(new FormData(dom.loginForm).get("email") || "").trim().toLowerCase();
      const password = String(new FormData(dom.loginForm).get("password") || "");

      const matchedUser = state.users.find((entry) => entry.email.toLowerCase() === email && entry.password === password);
      if (!matchedUser) {
        showToast("Invalid email or password.", "error");
        return;
      }

      state.session = {
        id: matchedUser.id,
        role: matchedUser.role
      };
      writeStore(STORAGE_KEYS.session, state.session);
      showToast("Logged in successfully.", "success");
      switchToApp();
    });

    dom.registerForm.addEventListener("submit", (event) => {
      event.preventDefault();
      const formData = new FormData(dom.registerForm);
      const name = String(formData.get("name") || "").trim();
      const email = String(formData.get("email") || "").trim().toLowerCase();
      const password = String(formData.get("password") || "");
      const role = String(formData.get("role") || "patient");

      if (state.users.some((entry) => entry.email.toLowerCase() === email)) {
        showToast("Email already exists.", "error");
        return;
      }

      const department = role === "doctor" ? DEPARTMENTS[Math.floor(Math.random() * DEPARTMENTS.length)] : "";
      const newUser = {
        id: uid("u"),
        name,
        email,
        password,
        role,
        department
      };

      state.users.push(newUser);
      writeStore(STORAGE_KEYS.users, state.users);
      state.session = { id: newUser.id, role: newUser.role };
      writeStore(STORAGE_KEYS.session, state.session);
      showToast("Account created and logged in.", "success");
      switchToApp();
    });

    dom.logoutBtn.addEventListener("click", () => {
      if (state.activeStreamController) {
        state.activeStreamController.abort();
      }
      localStorage.removeItem(STORAGE_KEYS.session);
      loadState();
      switchToAuth();
      showToast("Logged out.", "info");
    });

    dom.navButtons.forEach((btn) => {
      btn.addEventListener("click", () => {
        switchPanel(btn.dataset.panel);
      });
    });

    dom.chatForm.addEventListener("submit", handleChatSubmit);

    dom.themeMotionToggle.addEventListener("click", () => {
      const isReduced = document.body.classList.toggle("reduce-motion");
      localStorage.setItem(STORAGE_KEYS.reduceMotion, isReduced ? "1" : "0");
      dom.themeMotionToggle.textContent = isReduced ? "Enable Motion" : "Reduce Motion";
    });
  }

  function bindDom() {
    dom.authView = document.getElementById("authView");
    dom.appView = document.getElementById("appView");
    dom.loginTab = document.getElementById("loginTab");
    dom.registerTab = document.getElementById("registerTab");
    dom.loginForm = document.getElementById("loginForm");
    dom.registerForm = document.getElementById("registerForm");
    dom.logoutBtn = document.getElementById("logoutBtn");
    dom.userName = document.getElementById("userName");
    dom.userRole = document.getElementById("userRole");
    dom.userEmail = document.getElementById("userEmail");
    dom.workspacePanel = document.getElementById("workspacePanel");
    dom.assistantPanel = document.getElementById("assistantPanel");
    dom.analyticsPanel = document.getElementById("analyticsPanel");
    dom.chatForm = document.getElementById("chatForm");
    dom.chatInput = document.getElementById("chatInput");
    dom.chatSubmit = document.getElementById("chatSubmit");
    dom.chatStream = document.getElementById("chatStream");
    dom.themeMotionToggle = document.getElementById("themeMotionToggle");
    dom.navButtons = [...document.querySelectorAll(".nav-btn")];
  }

  function initMotionPreference() {
    const reduced = localStorage.getItem(STORAGE_KEYS.reduceMotion) === "1";
    document.body.classList.toggle("reduce-motion", reduced);
    dom.themeMotionToggle.textContent = reduced ? "Enable Motion" : "Reduce Motion";
  }

  function init() {
    bindDom();
    seedLocalData();
    loadState();
    initMotionPreference();
    attachEvents();
    applyAuthMode("login");

    if (state.session && getCurrentUser()) {
      switchToApp();
    } else {
      switchToAuth();
    }
  }

  document.addEventListener("DOMContentLoaded", init);
})();
