<?php
declare(strict_types=1);

/**
 * HIMS AI proxy
 * - Accepts POST JSON { message, context }
 * - Calls Groq OpenAI-compatible endpoint securely on the server
 * - Streams generated text back as SSE packets word-by-word
 */

header("X-Content-Type-Options: nosniff");
header("Referrer-Policy: same-origin");
header("X-Frame-Options: SAMEORIGIN");

if ($_SERVER["REQUEST_METHOD"] === "OPTIONS") {
    header("Allow: POST, OPTIONS");
    http_response_code(204);
    exit;
}

if ($_SERVER["REQUEST_METHOD"] !== "POST") {
    sendJsonError(405, "Method not allowed. Use POST.");
}

$contentType = $_SERVER["CONTENT_TYPE"] ?? "";
if (stripos($contentType, "application/json") === false) {
    sendJsonError(415, "Content-Type must be application/json.");
}

$rawBody = file_get_contents("php://input");
$payload = json_decode($rawBody ?: "", true);
if (!is_array($payload)) {
    sendJsonError(400, "Invalid JSON payload.");
}

$message = trim((string)($payload["message"] ?? ""));
if ($message === "") {
    sendJsonError(422, "message is required.");
}
$messageLength = function_exists("mb_strlen") ? mb_strlen($message) : strlen($message);
if ($messageLength > 6000) {
    sendJsonError(422, "message is too long.");
}

$clientIp = $_SERVER["REMOTE_ADDR"] ?? "unknown";
$rateError = rateLimitGuard($clientIp, 12, 60);
if ($rateError !== null) {
    sendJsonError(429, $rateError);
}

$context = is_array($payload["context"] ?? null) ? $payload["context"] : [];
$userRole = preg_replace("/[^a-z_]/i", "", (string)($context["role"] ?? "general")) ?: "general";
$userName = trim((string)($context["name"] ?? "Clinician"));
$mode = preg_replace("/[^a-z_]/i", "", (string)($context["mode"] ?? "interactive")) ?: "interactive";

$apiKey = getenv("GROQ_API_KEY") ?: getenv("AI_API_KEY");
if (!$apiKey) {
    $localConfig = __DIR__ . "/config.local.php";
    if (is_file($localConfig)) {
        require $localConfig;
    }
    $apiKey = getenv("GROQ_API_KEY") ?: getenv("AI_API_KEY");
}

if (!$apiKey) {
    sendJsonError(500, "Missing API key. Configure GROQ_API_KEY on server.");
}

$model = getenv("GROQ_MODEL") ?: "llama-3.3-70b-versatile";

$systemPrompt = implode("\n", [
    "You are HIMS-AI, a clinical assistant for hospital operations.",
    "Audience role: {$userRole}. Name context: {$userName}. Workflow mode: {$mode}.",
    "Style rules:",
    "- Use human, calm, practical language.",
    "- Use plain-text headings ending with a colon.",
    "- Never use markdown heading syntax (#) or bold markers (**).",
    "- Keep sections short and useful.",
    "- Avoid giving dangerous medical claims. Recommend professional validation when uncertain."
]);

$requestBody = [
    "model" => $model,
    "temperature" => 0.72,
    "max_tokens" => 900,
    "messages" => [
        ["role" => "system", "content" => $systemPrompt],
        ["role" => "user", "content" => $message]
    ]
];

$result = requestGroq($apiKey, $requestBody);
if (!$result["ok"]) {
    sendJsonError($result["status"], $result["error"]);
}

$aiText = trim((string)$result["text"]);
if ($aiText === "") {
    sendJsonError(502, "Model returned an empty response.");
}

streamAsSseWords($aiText);
exit;

function sendJsonError(int $status, string $message): void
{
    http_response_code($status);
    header("Content-Type: application/json; charset=utf-8");
    echo json_encode(["error" => $message], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
    exit;
}

function rateLimitGuard(string $ip, int $maxRequests, int $windowSeconds): ?string
{
    $bucketFile = sys_get_temp_dir() . "/hims_ai_rate_" . sha1($ip) . ".json";
    $now = time();

    $bucket = [
        "start" => $now,
        "count" => 0
    ];

    if (is_file($bucketFile)) {
        $raw = file_get_contents($bucketFile);
        $existing = json_decode($raw ?: "", true);
        if (is_array($existing) && isset($existing["start"], $existing["count"])) {
            $bucket["start"] = (int)$existing["start"];
            $bucket["count"] = (int)$existing["count"];
        }
    }

    if (($now - $bucket["start"]) >= $windowSeconds) {
        $bucket["start"] = $now;
        $bucket["count"] = 0;
    }

    if ($bucket["count"] >= $maxRequests) {
        return "Rate limit exceeded. Wait a minute and try again.";
    }

    $bucket["count"]++;
    file_put_contents($bucketFile, json_encode($bucket), LOCK_EX);
    return null;
}

function requestGroq(string $apiKey, array $payload): array
{
    $ch = curl_init("https://api.groq.com/openai/v1/chat/completions");
    if ($ch === false) {
        return [
            "ok" => false,
            "status" => 500,
            "error" => "Failed to initialize HTTP client."
        ];
    }

    curl_setopt_array($ch, [
        CURLOPT_POST => true,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_POSTFIELDS => json_encode($payload, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES),
        CURLOPT_HTTPHEADER => [
            "Authorization: Bearer " . $apiKey,
            "Content-Type: application/json"
        ],
        CURLOPT_TIMEOUT => 120,
        CURLOPT_CONNECTTIMEOUT => 20
    ]);

    $raw = curl_exec($ch);
    $status = (int)curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $curlError = curl_error($ch);
    curl_close($ch);

    if ($raw === false) {
        return [
            "ok" => false,
            "status" => 502,
            "error" => "Upstream request failed: " . ($curlError ?: "unknown error")
        ];
    }

    $decoded = json_decode($raw, true);
    if ($status >= 400) {
        $errorMessage = (string)($decoded["error"]["message"] ?? "Upstream AI request failed.");
        return [
            "ok" => false,
            "status" => $status,
            "error" => $errorMessage
        ];
    }

    $text = (string)($decoded["choices"][0]["message"]["content"] ?? "");
    return [
        "ok" => true,
        "status" => 200,
        "error" => "",
        "text" => $text
    ];
}

function streamAsSseWords(string $text): void
{
    @ini_set("zlib.output_compression", "0");
    @ini_set("output_buffering", "off");
    @ini_set("implicit_flush", "1");

    while (ob_get_level() > 0) {
        ob_end_flush();
    }
    ob_implicit_flush(true);

    header("Content-Type: text/event-stream; charset=utf-8");
    header("Cache-Control: no-cache, no-transform");
    header("Connection: keep-alive");
    header("X-Accel-Buffering: no");

    $tokens = preg_split("/(\s+)/u", $text, -1, PREG_SPLIT_DELIM_CAPTURE | PREG_SPLIT_NO_EMPTY);
    if (!is_array($tokens)) {
        $tokens = [$text];
    }

    foreach ($tokens as $token) {
        echo "data: " . json_encode(["word" => $token], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES) . "\n\n";
        @ob_flush();
        flush();
        usleep(32000);
    }

    echo "data: " . json_encode(["done" => true], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES) . "\n\n";
    @ob_flush();
    flush();
}
