document.addEventListener('DOMContentLoaded', () => {
    // --- DOM Elements ---
    const grid = document.getElementById('toolsGrid');
    const searchInput = document.getElementById('searchInput');
    const filterBtns = document.querySelectorAll('.filter-btn');
    const loader = document.getElementById('loader');

    // --- State ---
    let currentCategory = 'all';
    let searchQuery = '';

    // --- 1. LOADER ---
    window.addEventListener('load', () => {
        setTimeout(() => {
            loader.classList.add('hidden');
            // Trigger entry animations for cards after loader is gone
            setTimeout(renderTools, 100);
        }, 1500); // 1.5s artificial delay for the "system initialization" feel
    });

    // --- 2. RENDER TOOLS ---
    function renderTools() {
        grid.innerHTML = '';

        const filteredTools = aiTools.filter(tool => {
            const matchesCategory = currentCategory === 'all' || tool.category === currentCategory;
            const matchesSearch = tool.name.toLowerCase().includes(searchQuery) ||
                tool.description.toLowerCase().includes(searchQuery);
            return matchesCategory && matchesSearch;
        });

        if (filteredTools.length === 0) {
            grid.innerHTML = `<div style="text-align:center; grid-column: 1/-1; color: var(--text-dim); padding: 50px;">
                <i class="fa-solid fa-robot" style="font-size: 3rem; margin-bottom: 20px;"></i>
                <p>NO SYSTEMS FOUND MATCHING CRITERIA.</p>
            </div>`;
            return;
        }

        filteredTools.forEach((tool, index) => {
            const card = document.createElement('div');
            card.className = 'tool-card';
            card.style.animation = `fadeInUp 0.6s ease forwards ${index * 0.1}s`;
            card.style.opacity = '0'; // Start hidden for animation

            card.innerHTML = `
                <div class="card-icon-wrapper">
                    <i class="${tool.icon}"></i>
                </div>
                <div class="card-content">
                    <h3>${tool.name}</h3>
                    <span class="card-category">${tool.category.toUpperCase()}</span>
                    <p class="card-desc">${tool.description}</p>
                    <a href="${tool.url}" target="_blank" class="visit-btn">VISIT SYSTEM</a>
                </div>
            `;

            // Add slight 3D tilt effect on mousemove
            card.addEventListener('mousemove', handleTilt);
            card.addEventListener('mouseleave', resetTilt);

            grid.appendChild(card);
        });
    }

    // --- 3. FILTERING ---
    filterBtns.forEach(btn => {
        btn.addEventListener('click', () => {
            // Remove active class from all
            filterBtns.forEach(b => b.classList.remove('active'));
            // Add to clicked
            btn.classList.add('active');

            currentCategory = btn.getAttribute('data-category');
            renderTools();
        });
    });

    // --- 4. SEARCH ---
    searchInput.addEventListener('input', (e) => {
        searchQuery = e.target.value.toLowerCase();
        renderTools();
    });

    // --- 5. TILT EFFECT ---
    function handleTilt(e) {
        const card = e.currentTarget;
        const rect = card.getBoundingClientRect();
        const x = e.clientX - rect.left;
        const y = e.clientY - rect.top;

        const centerX = rect.width / 2;
        const centerY = rect.height / 2;

        const rotateX = ((y - centerY) / centerY) * -5; // Max 5deg rotation
        const rotateY = ((x - centerX) / centerX) * 5;

        card.style.transform = `perspective(1000px) rotateX(${rotateX}deg) rotateY(${rotateY}deg) scale(1.02)`;
    }

    function resetTilt(e) {
        const card = e.currentTarget;
        card.style.transform = 'perspective(1000px) rotateX(0) rotateY(0) scale(1)';
    }

    // --- CSS Animations Injection ---
    // Inject the keyframes for card entry if not in CSS
    const styleSheet = document.createElement("style");
    styleSheet.innerText = `
        @keyframes fadeInUp {
            from {
                opacity: 0;
                transform: translateY(30px);
            }
            to {
                opacity: 1;
                transform: translateY(0);
            }
        }
    `;
    document.head.appendChild(styleSheet);
});
