<?php
require_once '../config/database.php';
require_once 'verify-browser.php';

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    die(json_encode(['error' => 'Method not allowed']));
}

$data = json_decode(file_get_contents('php://input'), true);
$session_token = $data['session_token'] ?? '';

if (empty($session_token)) {
    http_response_code(400);
    die(json_encode(['error' => 'Session token required']));
}

try {
    $pdo->beginTransaction();

    // Mark session as submitted
    $stmt = $pdo->prepare("
        UPDATE active_sessions 
        SET session_status = 'submitted',
            submitted_at = NOW()
        WHERE session_token = ? AND session_status = 'active'
    ");
    $stmt->execute([$session_token]);

    // Get session details
    $stmt = $pdo->prepare("
        SELECT s.learner_id, s.assessment_id, s.session_status
        FROM active_sessions s
        WHERE s.session_token = ?
    ");
    $stmt->execute([$session_token]);
    $session = $stmt->fetch();

    if (!$session) {
        throw new Exception('Session not found');
    }

    // Calculate score
    $stmt = $pdo->prepare("
        SELECT 
            SUM(CASE WHEN lr.selected_choice = qb.correct_choice THEN qb.point_value ELSE 0 END) as earned_score,
            SUM(qb.point_value) as max_score
        FROM learner_responses lr
        JOIN question_bank qb ON lr.question_id = qb.question_id
        WHERE lr.session_token = ?
    ");
    $stmt->execute([$session_token]);
    $score_data = $stmt->fetch();

    $earned = $score_data['earned_score'] ?? 0;
    $maximum = $score_data['max_score'] ?? 1;

    // Get passing threshold
    $stmt = $pdo->prepare("SELECT passing_threshold FROM assessments WHERE assessment_id = ?");
    $stmt->execute([$session['assessment_id']]);
    $threshold = $stmt->fetchColumn() ?? 50;

    $percentage = ($earned / $maximum) * 100;
    $passed = $percentage >= $threshold;

    // Insert result
    $stmt = $pdo->prepare("
        INSERT INTO assessment_results (
            session_token, learner_id, assessment_id, 
            total_score, maximum_score, pass_status, was_terminated
        ) VALUES (?, ?, ?, ?, ?, ?, ?)
    ");

    $was_terminated = $session['session_status'] === 'terminated';
    $stmt->execute([
        $session_token,
        $session['learner_id'],
        $session['assessment_id'],
        $earned,
        $maximum,
        $passed,
        $was_terminated
    ]);

    $pdo->commit();

    echo json_encode([
        'success' => true,
        'result' => [
            'score' => $earned,
            'max_score' => $maximum,
            'percentage' => round($percentage, 2),
            'passed' => $passed,
            'terminated' => $was_terminated
        ]
    ]);

} catch (Exception $e) {
    $pdo->rollBack();
    http_response_code(500);
    echo json_encode(['error' => 'Failed to submit exam', 'details' => $e->getMessage()]);
}
?>
