<?php
/**
 * Get All Progress API
 * Returns all student exam progress for admin dashboard
 */

require_once '../config/database.php';

header('Content-Type: application/json');

// Allow GET and POST
if ($_SERVER['REQUEST_METHOD'] !== 'GET' && $_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    die(json_encode(['error' => 'Method not allowed']));
}

try {
    // Get all progress records with exam details
    $stmt = $pdo->prepare("
        SELECT 
            ep.id,
            ep.session_token,
            ep.student_email,
            ep.student_name,
            ep.roll_no,
            ep.exam_code,
            ep.answers_json,
            ep.time_remaining,
            ep.current_question,
            ep.total_questions,
            ep.answered_count,
            ep.status as progress_status,
            ep.last_sync,
            ep.created_at,
            a.assessment_title as exam_title,
            a.total_duration_mins,
            s.session_status,
            s.termination_reason,
            s.violation_count,
            s.submitted_at,
            ar.total_score,
            ar.maximum_score,
            ar.pass_status
        FROM exam_progress ep
        LEFT JOIN assessments a ON ep.exam_code = a.exam_code
        LEFT JOIN active_sessions s ON ep.session_token = s.session_token
        LEFT JOIN assessment_results ar ON ep.session_token = ar.session_token
        ORDER BY ep.last_sync DESC
    ");
    $stmt->execute();
    $progress = $stmt->fetchAll();

    // Also get question details for answer mapping
    $questions_by_exam = [];
    $stmt = $pdo->prepare("
        SELECT 
            a.exam_code,
            q.question_id,
            q.question_order,
            q.question_prompt,
            q.choice_a,
            q.choice_b,
            q.choice_c,
            q.choice_d,
            q.correct_choice
        FROM question_bank q
        JOIN assessments a ON q.assessment_id = a.assessment_id
        ORDER BY q.question_order
    ");
    $stmt->execute();
    $questions = $stmt->fetchAll();
    
    foreach ($questions as $q) {
        $code = $q['exam_code'];
        if (!isset($questions_by_exam[$code])) {
            $questions_by_exam[$code] = [];
        }
        $questions_by_exam[$code][$q['question_id']] = $q;
    }

    // Format response
    $formatted = [];
    foreach ($progress as $p) {
        $answers = json_decode($p['answers_json'], true) ?: [];
        $exam_questions = $questions_by_exam[$p['exam_code']] ?? [];
        
        // Map answers to question details
        $detailed_answers = [];
        foreach ($answers as $qid => $selected) {
            $q = $exam_questions[$qid] ?? null;
            if ($q) {
                $detailed_answers[] = [
                    'question_id' => $qid,
                    'question_order' => $q['question_order'],
                    'question_prompt' => $q['question_prompt'],
                    'selected' => $selected,
                    'correct' => $q['correct_choice'],
                    'is_correct' => in_array($q['correct_choice'], (array)$selected)
                ];
            }
        }

        $formatted[] = [
            'id' => $p['id'],
            'student' => [
                'name' => $p['student_name'],
                'email' => $p['student_email'],
                'roll_no' => $p['roll_no'] ?? ''
            ],
            'exam' => [
                'code' => $p['exam_code'],
                'title' => $p['exam_title'] ?? $p['exam_code'],
                'duration_minutes' => $p['total_duration_mins']
            ],
            'progress' => [
                'current_question' => $p['current_question'],
                'total_questions' => $p['total_questions'],
                'answered_count' => $p['answered_count'],
                'time_remaining' => $p['time_remaining'],
                'percentage' => $p['total_questions'] > 0 
                    ? round(($p['answered_count'] / $p['total_questions']) * 100) 
                    : 0
            ],
            'status' => $p['session_status'] ?? $p['progress_status'] ?? 'active',
            'termination_reason' => $p['termination_reason'] ?? null,
            'violation_count' => $p['violation_count'] ?? 0,
            'submitted_at' => $p['submitted_at'],
            'result' => $p['total_score'] !== null ? [
                'score' => $p['total_score'],
                'max_score' => $p['maximum_score'],
                'passed' => (bool)$p['pass_status']
            ] : null,
            'answers' => $detailed_answers,
            'last_sync' => $p['last_sync'],
            'started_at' => $p['created_at']
        ];
    }

    echo json_encode([
        'success' => true,
        'count' => count($formatted),
        'students' => $formatted
    ]);

} catch (PDOException $e) {
    http_response_code(500);
    echo json_encode(['error' => 'Database error', 'details' => $e->getMessage()]);
}
?>
