<?php
// CORS Headers - Allow requests from exam page
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');
header('Content-Type: application/json');

// Handle preflight requests
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

require_once '../config/database.php';

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    die(json_encode(['error' => 'Method not allowed']));
}

$data = json_decode(file_get_contents('php://input'), true);

$session_token = $data['session_token'] ?? '';
$question_id = $data['question_id'] ?? 0;
$selected_choice = $data['selected_choice'] ?? '';
$response_time = $data['response_time_seconds'] ?? 0;

// Debug log
error_log("submit-answer.php: session=$session_token, qid=$question_id, choice=$selected_choice");

if (empty($session_token) || empty($question_id)) {
    http_response_code(400);
    die(json_encode(['error' => 'Missing required fields', 'debug' => [
        'session_token' => !empty($session_token),
        'question_id' => !empty($question_id),
        'selected_choice' => $selected_choice
    ]]));
}

// Allow empty selected_choice (for clearing an answer)
if (empty($selected_choice)) {
    $selected_choice = '';
}

try {
    // First verify the session exists
    $checkSession = $pdo->prepare("SELECT session_token FROM active_sessions WHERE session_token = ?");
    $checkSession->execute([$session_token]);
    if (!$checkSession->fetch()) {
        http_response_code(404);
        die(json_encode(['error' => 'Session not found', 'session_token' => substr($session_token, 0, 20) . '...']));
    }

    // Insert or update answer
    $stmt = $pdo->prepare("
        INSERT INTO learner_responses (session_token, question_id, selected_choice, response_time_seconds, answered_at)
        VALUES (?, ?, ?, ?, NOW())
        ON DUPLICATE KEY UPDATE 
            selected_choice = VALUES(selected_choice),
            response_time_seconds = VALUES(response_time_seconds),
            answered_at = NOW()
    ");

    $stmt->execute([$session_token, $question_id, $selected_choice, $response_time]);

    echo json_encode([
        'success' => true,
        'message' => 'Answer saved',
        'question_id' => $question_id
    ]);

} catch (PDOException $e) {
    error_log("submit-answer.php error: " . $e->getMessage());
    http_response_code(500);
    echo json_encode(['error' => 'Failed to save answer', 'details' => $e->getMessage()]);
}
?>
