<?php
// Auto-setup and fix database for BGNU Exam System
// Access this file once to fix all database issues
header('Content-Type: text/html; charset=utf-8');

require_once '../config/database.php';

$messages = [];
$errors = [];

function logMessage($msg) {
    global $messages;
    $messages[] = $msg;
}

function logError($msg) {
    global $errors;
    $errors[] = $msg;
}

try {
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    
    // 1. Create flutter_questions table if not exists
    logMessage("Step 1: Creating flutter_questions table...");
    $pdo->exec("
        CREATE TABLE IF NOT EXISTS flutter_questions (
            id INT AUTO_INCREMENT PRIMARY KEY,
            title VARCHAR(500) NOT NULL,
            option_a VARCHAR(255) NOT NULL,
            option_b VARCHAR(255) NOT NULL,
            option_c VARCHAR(255) NOT NULL,
            option_d VARCHAR(255) NOT NULL,
            correct_answers VARCHAR(50) NOT NULL,
            is_multi_answer BOOLEAN DEFAULT FALSE,
            point_value DECIMAL(5,2) DEFAULT 1.00,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4
    ");
    
    // 2. Check if flutter_questions has data
    $stmt = $pdo->query("SELECT COUNT(*) FROM flutter_questions");
    $count = $stmt->fetchColumn();
    
    if ($count < 30) {
        logMessage("Step 2: Found $count questions. Clearing and inserting 30 questions...");
        $pdo->exec("DELETE FROM flutter_questions");
        
        $questions = [
            ['What programming language is used to build Flutter apps?', 'Java', 'Dart', 'Python', 'Swift', 'B', 0],
            ['Which widget is used to display text in Flutter?', 'Text', 'Label', 'Paragraph', 'TextBox', 'A', 0],
            ['What is the main function in a Flutter app called?', 'start()', 'main()', 'run()', 'init()', 'B', 0],
            ['Which widget creates a button in Flutter?', 'ElevatedButton', 'Button', 'Clickable', 'TouchableButton', 'A', 0],
            ['What widget is used to arrange children vertically?', 'Row', 'Column', 'Stack', 'Horizontal', 'B', 0],
            ['What widget is used to arrange children horizontally?', 'Row', 'Column', 'Stack', 'Vertical', 'A', 0],
            ['Which command creates a new Flutter project?', 'flutter new', 'flutter create', 'flutter init', 'flutter start', 'B', 0],
            ['What is the root widget of most Flutter apps?', 'MainApp', 'MaterialApp', 'FlutterApp', 'RootWidget', 'B', 0],
            ['Which widget adds padding around its child?', 'Margin', 'Padding', 'Spacing', 'Border', 'B', 0],
            ['What file contains Flutter dependencies?', 'package.json', 'build.gradle', 'pubspec.yaml', 'dependencies.txt', 'C', 0],
            ['Which widget displays an image in Flutter?', 'Picture', 'Image', 'Photo', 'ImageView', 'B', 0],
            ['What does hot reload do in Flutter?', 'Restarts app completely', 'Updates UI without losing state', 'Clears all data', 'Compiles code', 'B', 0],
            ['Which widget creates a scrollable list?', 'ScrollView', 'ListView', 'ListBox', 'ScrollList', 'B', 0],
            ['What widget creates a text input field?', 'TextInput', 'InputField', 'TextField', 'EditText', 'C', 0],
            ['Which widgets add space between widgets? (Select ALL)', 'SizedBox', 'Spacer', 'Gap', 'Divider', 'A,B', 1],
            ['What is a Scaffold in Flutter?', 'A button', 'A material design layout', 'A database', 'An animation', 'B', 0],
            ['Which widget is used for navigation?', 'Navigator', 'Router', 'PageController', 'ScreenSwitcher', 'A', 0],
            ['What are the two types of widgets?', 'Stateless and Stateful', 'Smart and Dumb', 'Active and Passive', 'Live and Dead', 'A', 0],
            ['Which command runs a Flutter app?', 'flutter start', 'flutter run', 'flutter execute', 'flutter launch', 'B', 0],
            ['What shows a circular loading indicator?', 'LoadingSpinner', 'CircularProgressIndicator', 'ProgressCircle', 'LoadingCircle', 'B', 0],
            ['Which widgets make child clickable? (Select ALL)', 'GestureDetector', 'InkWell', 'Touchable', 'Clickable', 'A,B', 1],
            ['What does setState() do?', 'Saves to database', 'Updates UI when state changes', 'Closes app', 'Creates widget', 'B', 0],
            ['Which widget displays icons?', 'Icon', 'Symbol', 'Glyph', 'Emoji', 'A', 0],
            ['What is BuildContext used for?', 'Building the app', 'Locating widgets in tree', 'Compiling code', 'Internet connection', 'B', 0],
            ['Which creates a floating action button?', 'FloatingActionButton', 'FAB', 'ActionButton', 'HoverButton', 'A', 0],
            ['What is the extension for Dart files?', '.flutter', '.dart', '.ft', '.dr', 'B', 0],
            ['Which widget shows a popup dialog?', 'AlertDialog', 'PopupWindow', 'DialogBox', 'ModalWindow', 'A', 0],
            ['What method runs when StatefulWidget is created?', 'onCreate()', 'initState()', 'start()', 'initialize()', 'B', 0],
            ['Which widget creates a grid layout?', 'GridView', 'GridLayout', 'TableView', 'MatrixView', 'A', 0],
            ['What is Flutter primarily used for?', 'Backend', 'Cross-platform mobile apps', 'Games only', 'Databases', 'B', 0],
        ];
        
        $stmt = $pdo->prepare("INSERT INTO flutter_questions (title, option_a, option_b, option_c, option_d, correct_answers, is_multi_answer) VALUES (?, ?, ?, ?, ?, ?, ?)");
        foreach ($questions as $q) {
            $stmt->execute($q);
        }
        logMessage("Inserted 30 questions successfully!");
    } else {
        logMessage("Step 2: Already have $count questions. Skipping insertion.");
    }
    
    // 3. Fix learner_responses table - remove FK and fix column
    logMessage("Step 3: Fixing learner_responses table...");
    
    // Check for foreign keys
    $stmt = $pdo->query("
        SELECT CONSTRAINT_NAME 
        FROM information_schema.KEY_COLUMN_USAGE 
        WHERE TABLE_SCHEMA = DATABASE() 
        AND TABLE_NAME = 'learner_responses' 
        AND REFERENCED_TABLE_NAME IS NOT NULL
    ");
    $fks = $stmt->fetchAll(PDO::FETCH_COLUMN);
    
    $pdo->exec("SET FOREIGN_KEY_CHECKS = 0");
    foreach ($fks as $fk) {
        try {
            $pdo->exec("ALTER TABLE learner_responses DROP FOREIGN KEY `$fk`");
            logMessage("Dropped foreign key: $fk");
        } catch (Exception $e) {
            // Ignore if already dropped
        }
    }
    $pdo->exec("SET FOREIGN_KEY_CHECKS = 1");
    
    // Fix column size
    try {
        $pdo->exec("ALTER TABLE learner_responses MODIFY COLUMN selected_choice VARCHAR(20)");
        logMessage("Modified selected_choice column to VARCHAR(20)");
    } catch (Exception $e) {
        logMessage("Column already correct or: " . $e->getMessage());
    }
    
    // 4. Create/update FLUTTER2026 assessment
    logMessage("Step 4: Creating/updating FLUTTER2026 assessment...");
    
    // Get an admin learner_id
    $stmt = $pdo->query("SELECT learner_id FROM learners LIMIT 1");
    $learner = $stmt->fetch();
    $admin_id = $learner ? $learner['learner_id'] : 1;
    
    // Check if assessment exists
    $stmt = $pdo->prepare("SELECT assessment_id FROM assessments WHERE exam_code = 'FLUTTER2026'");
    $stmt->execute();
    $existing = $stmt->fetch();
    
    if ($existing) {
        $pdo->exec("UPDATE assessments SET question_count = 30, assessment_status = 'active' WHERE exam_code = 'FLUTTER2026'");
        logMessage("Updated FLUTTER2026 assessment to 30 questions and active status");
    } else {
        $stmt = $pdo->prepare("
            INSERT INTO assessments (exam_code, assessment_title, created_by, total_duration_mins, question_count, scheduled_start, scheduled_end, passing_threshold, assessment_status)
            VALUES ('FLUTTER2026', 'Flutter Basics Exam', ?, 60, 30, NOW(), DATE_ADD(NOW(), INTERVAL 1 YEAR), 50.00, 'active')
        ");
        $stmt->execute([$admin_id]);
        logMessage("Created new FLUTTER2026 assessment");
    }
    
    // 5. Clear stale sessions
    logMessage("Step 5: Clearing stale/terminated sessions...");
    $pdo->exec("DELETE FROM active_sessions WHERE session_status IN ('terminated', 'abandoned')");
    $pdo->exec("DELETE FROM active_sessions WHERE session_started_at < DATE_SUB(NOW(), INTERVAL 12 HOUR)");
    logMessage("Cleared old sessions");
    
    // Final verification
    logMessage("=== VERIFICATION ===");
    $stmt = $pdo->query("SELECT COUNT(*) FROM flutter_questions");
    logMessage("Flutter questions: " . $stmt->fetchColumn());
    
    $stmt = $pdo->query("SELECT question_count, assessment_status FROM assessments WHERE exam_code = 'FLUTTER2026'");
    $ass = $stmt->fetch();
    logMessage("Assessment: question_count=" . ($ass['question_count'] ?? 'N/A') . ", status=" . ($ass['assessment_status'] ?? 'N/A'));
    
} catch (Exception $e) {
    logError("Error: " . $e->getMessage());
}

?>
<!DOCTYPE html>
<html>
<head>
    <title>BGNU Exam Database Setup</title>
    <style>
        body { font-family: monospace; background: #0d1117; color: #c9d1d9; padding: 40px; }
        .success { color: #3fb950; }
        .error { color: #f85149; }
        h1 { color: #58a6ff; }
        .message { margin: 8px 0; padding: 8px; background: #161b22; border-radius: 4px; }
    </style>
</head>
<body>
    <h1>🔧 BGNU Exam Database Setup</h1>
    
    <?php if (empty($errors)): ?>
        <p class="success">✅ Setup completed successfully!</p>
    <?php else: ?>
        <p class="error">❌ Some errors occurred:</p>
        <?php foreach ($errors as $err): ?>
            <div class="message error"><?= htmlspecialchars($err) ?></div>
        <?php endforeach; ?>
    <?php endif; ?>
    
    <h2>Setup Log:</h2>
    <?php foreach ($messages as $msg): ?>
        <div class="message"><?= htmlspecialchars($msg) ?></div>
    <?php endforeach; ?>
    
    <br><br>
    <a href="../exam/exam.php" style="color: #58a6ff;">→ Go to Exam Page</a>
</body>
</html>
