<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Live Exam Monitor - BGNU Exam System</title>
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', sans-serif;
            background: #0D1117;
            color: #fff;
            padding: 20px;
        }
        .header {
            background: #161B22;
            padding: 20px;
            border-radius: 10px;
            margin-bottom: 20px;
        }
        h1 { color: #00FFA3; margin-bottom: 10px; }
        .stats {
            display: flex;
            gap: 20px;
            margin-top: 15px;
        }
        .stat-card {
            background: #1C2128;
            padding: 15px 25px;
            border-radius: 8px;
            border-left: 4px solid #00FFA3;
        }
        .stat-number { font-size: 32px; font-weight: bold; }
        .stat-label { color: #8B949E; font-size: 14px; }
        
        .student-grid {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(350px, 1fr));
            gap: 15px;
        }
        .student-card {
            background: #161B22;
            border-radius: 10px;
            padding: 20px;
            border: 2px solid transparent;
            transition: all 0.3s;
        }
        .student-card.active { border-color: #00FFA3; }
        .student-card.warning { border-color: #FFB703; }
        .student-card.terminated { border-color: #FF6B6B; opacity: 0.6; }
        
        .student-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 15px;
        }
        .student-id { font-size: 18px; font-weight: bold; }
        .status-badge {
            padding: 5px 12px;
            border-radius: 20px;
            font-size: 12px;
            font-weight: bold;
        }
        .progress-bar {
            background: #1C2128;
            height: 25px;
            border-radius: 12px;
            overflow: hidden;
            margin: 10px 0;
        }
        .progress-fill {
            height: 100%;
            background: linear-gradient(90deg, #00FFA3, #4CC9F0);
            transition: width 0.5s;
            display: flex;
            align-items: center;
            padding-left: 10px;
            font-size: 12px;
            font-weight: bold;
        }
        .info-row {
            display: flex;
            justify-content: space-between;
            margin: 8px 0;
            font-size: 14px;
        }
        .info-label { color: #8B949E; }
        .time-remaining { color: #FFB703; font-weight: bold; }
        
        #lastUpdate {
            text-align: center;
            color: #8B949E;
            margin-top: 20px;
            font-size: 12px;
        }
    </style>
</head>
<body>
    <div class="header">
        <h1>🎓 Live Exam Monitor</h1>
        <p style="color: #8B949E;">Real-time student tracking with instant cheating detection</p>
        <div class="stats">
            <div class="stat-card">
                <div class="stat-number" id="totalActive">0</div>
                <div class="stat-label">Active Students</div>
            </div>
            <div class="stat-card">
                <div class="stat-number" id="totalWarnings">0</div>
                <div class="stat-label">Warnings</div>
            </div>
            <div class="stat-card">
                <div class="stat-number" id="totalTerminated">0</div>
                <div class="stat-label">Terminated</div>
            </div>
        </div>
    </div>

    <div class="student-grid" id="studentGrid"></div>
    <div id="lastUpdate">Loading...</div>

    <script>
        function refreshDashboard() {
            fetch('../api/get-live-sessions.php')
                .then(res => res.json())
                .then(data => {
                    if (!data.success) {
                        console.error('Failed to fetch sessions');
                        return;
                    }

                    const grid = document.getElementById('studentGrid');
                    let activeCount = 0;
                    let warningCount = 0;
                    let terminatedCount = 0;

                    grid.innerHTML = data.sessions.map(student => {
                        // Count statuses
                        if (student.status_code === 'active') activeCount++;
                        else if (student.status_code === 'warning') warningCount++;
                        else if (student.status_code === 'terminated') terminatedCount++;

                        // Calculate progress percentage
                        const progress = parseInt(student.progress.split(' / ')[0]);
                        const total = parseInt(student.progress.split(' / ')[1]);
                        const percentage = Math.round((progress / total) * 100);

                        // Status badge color
                        let badgeStyle = 'background: #00FFA3';
                        if (student.status_code === 'warning') badgeStyle = 'background: #FFB703';
                        if (student.status_code === 'terminated') badgeStyle = 'background: #FF6B6B';

                        return `
                            <div class="student-card ${student.status_code}">
                                <div class="student-header">
                                    <div class="student-id">${student.student_code}</div>
                                    <div class="status-badge" style="${badgeStyle}">${student.status}</div>
                                </div>
                                <div style="color: #C9D1D9; margin-bottom: 10px;">${student.student_name}</div>
                                
                                <div class="progress-bar">
                                    <div class="progress-fill" style="width: ${percentage}%">
                                        ${student.progress}
                                    </div>
                                </div>

                                <div class="info-row">
                                    <span class="info-label">Time Remaining:</span>
                                    <span class="time-remaining">${student.time_left}</span>
                                </div>
                                <div class="info-row">
                                    <span class="info-label">Window Status:</span>
                                    <span>${student.window_status}</span>
                                </div>
                                <div class="info-row">
                                    <span class="info-label">Network:</span>
                                    <span>${student.network}</span>
                                </div>
                                <div class="info-row">
                                    <span class="info-label">Last update:</span>
                                    <span>${student.last_ping_seconds_ago}s ago</span>
                                </div>
                            </div>
                        `;
                    }).join('');

                    // Update stats
                    document.getElementById('totalActive').textContent = activeCount;
                    document.getElementById('totalWarnings').textContent = warningCount;
                    document.getElementById('totalTerminated').textContent = terminatedCount;
                    document.getElementById('lastUpdate').textContent = `Last updated: ${data.timestamp}`;
                })
                .catch(err => {
                    console.error('Error fetching data:', err);
                    document.getElementById('lastUpdate').textContent = 'Error loading data';
                });
        }

        // Refresh every 3 seconds for instant updates
        refreshDashboard();
        setInterval(refreshDashboard, 3000);
    </script>
</body>
</html>
