<?php
/**
 * Dino Browser - History Sync API
 * Handles browsing history synchronization
 */

require_once 'config.php';

$input = getInput();
$action = $input['action'] ?? $_GET['action'] ?? '';

switch ($action) {
    case 'get':
        handleGet();
        break;
    case 'sync':
        handleSync($input);
        break;
    case 'clear':
        handleClear($input);
        break;
    default:
        jsonResponse(['success' => false, 'error' => 'Invalid action'], 400);
}

/**
 * Get history for user
 */
function handleGet() {
    $userId = sanitize($_GET['user_id'] ?? '');
    $limit = intval($_GET['limit'] ?? 500);
    $offset = intval($_GET['offset'] ?? 0);
    $from = $_GET['from'] ?? null;
    $to = $_GET['to'] ?? null;
    
    if (empty($userId)) {
        jsonResponse(['success' => false, 'error' => 'Missing user_id'], 400);
    }
    
    $limit = min($limit, 1000); // Max 1000 entries
    
    $pdo = getConnection();
    
    try {
        $sql = "
            SELECT id, url, title, favicon_url, visited_at, visit_count
            FROM history 
            WHERE user_id = ?
        ";
        $params = [$userId];
        
        if ($from) {
            $sql .= " AND visited_at >= ?";
            $params[] = $from;
        }
        
        if ($to) {
            $sql .= " AND visited_at <= ?";
            $params[] = $to;
        }
        
        $sql .= " ORDER BY visited_at DESC LIMIT ? OFFSET ?";
        $params[] = $limit;
        $params[] = $offset;
        
        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);
        $history = $stmt->fetchAll();
        
        // Get total count
        $stmt = $pdo->prepare("SELECT COUNT(*) as total FROM history WHERE user_id = ?");
        $stmt->execute([$userId]);
        $total = $stmt->fetch()['total'];
        
        logSync($pdo, $userId, 'history', count($history), 'download', 'success');
        
        jsonResponse([
            'success' => true,
            'history' => $history,
            'total' => intval($total),
            'hasMore' => ($offset + count($history)) < $total
        ]);
        
    } catch (PDOException $e) {
        jsonResponse(['success' => false, 'error' => 'Failed to fetch history'], 500);
    }
}

/**
 * Sync history from client
 */
function handleSync($input) {
    if (!validateRequired($input, ['user_id', 'history'])) {
        jsonResponse(['success' => false, 'error' => 'Missing required fields'], 400);
    }
    
    $userId = sanitize($input['user_id']);
    $historyEntries = $input['history'];
    
    $pdo = getConnection();
    
    try {
        $pdo->beginTransaction();
        $syncedCount = 0;
        
        foreach ($historyEntries as $entry) {
            $entryId = sanitize($entry['id'] ?? '');
            if (empty($entryId)) continue;
            
            $stmt = $pdo->prepare("
                INSERT INTO history (id, user_id, url, title, favicon_url, visited_at, visit_count)
                VALUES (?, ?, ?, ?, ?, ?, ?)
                ON DUPLICATE KEY UPDATE
                    title = VALUES(title),
                    favicon_url = VALUES(favicon_url),
                    visit_count = visit_count + 1,
                    visited_at = GREATEST(visited_at, VALUES(visited_at))
            ");
            
            $stmt->execute([
                $entryId,
                $userId,
                $entry['url'] ?? '',
                sanitize($entry['title'] ?? ''),
                $entry['faviconUrl'] ?? null,
                $entry['visitedAt'] ?? date('Y-m-d H:i:s'),
                intval($entry['visitCount'] ?? 1)
            ]);
            
            $syncedCount++;
        }
        
        $pdo->commit();
        
        logSync($pdo, $userId, 'history', $syncedCount, 'upload', 'success');
        
        jsonResponse([
            'success' => true,
            'synced' => $syncedCount
        ]);
        
    } catch (PDOException $e) {
        $pdo->rollBack();
        logSync($pdo, $userId, 'history', 0, 'upload', 'failed', $e->getMessage());
        jsonResponse(['success' => false, 'error' => 'Sync failed'], 500);
    }
}

/**
 * Clear history for user
 */
function handleClear($input) {
    if (!validateRequired($input, ['user_id'])) {
        jsonResponse(['success' => false, 'error' => 'Missing user_id'], 400);
    }
    
    $userId = sanitize($input['user_id']);
    $from = $input['from'] ?? null;
    $to = $input['to'] ?? null;
    
    $pdo = getConnection();
    
    try {
        if ($from && $to) {
            // Clear specific date range
            $stmt = $pdo->prepare("
                DELETE FROM history 
                WHERE user_id = ? AND visited_at >= ? AND visited_at <= ?
            ");
            $stmt->execute([$userId, $from, $to]);
        } else {
            // Clear all
            $stmt = $pdo->prepare("DELETE FROM history WHERE user_id = ?");
            $stmt->execute([$userId]);
        }
        
        jsonResponse([
            'success' => true,
            'cleared' => $stmt->rowCount()
        ]);
        
    } catch (PDOException $e) {
        jsonResponse(['success' => false, 'error' => 'Clear failed'], 500);
    }
}
