<?php
/**
 * Dino Browser - Authentication API
 * Handles user registration and authentication
 */

require_once 'config.php';

$input = getInput();
$action = $input['action'] ?? '';

switch ($action) {
    case 'register':
        handleRegister($input);
        break;
    case 'login':
        handleLogin($input);
        break;
    case 'delete':
        handleDelete($input);
        break;
    default:
        jsonResponse(['success' => false, 'error' => 'Invalid action'], 400);
}

/**
 * Register new user
 */
function handleRegister($input) {
    if (!validateRequired($input, ['firebase_uid', 'email'])) {
        jsonResponse(['success' => false, 'error' => 'Missing required fields'], 400);
    }
    
    $firebaseUid = sanitize($input['firebase_uid']);
    $email = sanitize($input['email']);
    $displayName = isset($input['display_name']) ? sanitize($input['display_name']) : null;
    
    $pdo = getConnection();
    
    try {
        // Check if user already exists
        $stmt = $pdo->prepare("SELECT id FROM users WHERE firebase_uid = ?");
        $stmt->execute([$firebaseUid]);
        
        if ($stmt->fetch()) {
            // User exists, update last sync
            $stmt = $pdo->prepare("UPDATE users SET last_sync_at = NOW(), email = ? WHERE firebase_uid = ?");
            $stmt->execute([$email, $firebaseUid]);
            
            jsonResponse(['success' => true, 'message' => 'User updated']);
        }
        
        // Insert new user
        $stmt = $pdo->prepare("
            INSERT INTO users (firebase_uid, email, display_name, created_at)
            VALUES (?, ?, ?, NOW())
        ");
        $stmt->execute([$firebaseUid, $email, $displayName]);
        
        logSync($pdo, $firebaseUid, 'full', 0, 'upload', 'success');
        
        jsonResponse(['success' => true, 'message' => 'User registered']);
        
    } catch (PDOException $e) {
        jsonResponse(['success' => false, 'error' => 'Registration failed'], 500);
    }
}

/**
 * Login user (verify and update last sync)
 */
function handleLogin($input) {
    if (!validateRequired($input, ['firebase_uid'])) {
        jsonResponse(['success' => false, 'error' => 'Missing firebase_uid'], 400);
    }
    
    $firebaseUid = sanitize($input['firebase_uid']);
    
    $pdo = getConnection();
    
    try {
        $stmt = $pdo->prepare("
            UPDATE users SET last_sync_at = NOW() 
            WHERE firebase_uid = ? AND is_active = 1
        ");
        $stmt->execute([$firebaseUid]);
        
        if ($stmt->rowCount() > 0) {
            jsonResponse(['success' => true, 'message' => 'Login successful']);
        } else {
            jsonResponse(['success' => false, 'error' => 'User not found'], 404);
        }
        
    } catch (PDOException $e) {
        jsonResponse(['success' => false, 'error' => 'Login failed'], 500);
    }
}

/**
 * Delete user and all their data
 */
function handleDelete($input) {
    if (!validateRequired($input, ['user_id'])) {
        jsonResponse(['success' => false, 'error' => 'Missing user_id'], 400);
    }
    
    $userId = sanitize($input['user_id']);
    
    $pdo = getConnection();
    
    try {
        $pdo->beginTransaction();
        
        // Delete all user data
        $tables = ['bookmarks', 'bookmark_folders', 'history', 'summaries', 'extensions', 'settings', 'sync_log'];
        
        foreach ($tables as $table) {
            $stmt = $pdo->prepare("DELETE FROM `$table` WHERE user_id = ?");
            $stmt->execute([$userId]);
        }
        
        // Delete user
        $stmt = $pdo->prepare("DELETE FROM users WHERE firebase_uid = ?");
        $stmt->execute([$userId]);
        
        $pdo->commit();
        
        jsonResponse(['success' => true, 'message' => 'User and data deleted']);
        
    } catch (PDOException $e) {
        $pdo->rollBack();
        jsonResponse(['success' => false, 'error' => 'Delete failed'], 500);
    }
}
