<?php
/**
 * Dino Browser - Extensions Sync API
 * Handles extension metadata synchronization (not actual extension files)
 */

require_once 'config.php';

$input = getInput();
$action = $input['action'] ?? $_GET['action'] ?? '';

switch ($action) {
    case 'get':
        handleGet();
        break;
    case 'sync':
        handleSync($input);
        break;
    case 'toggle':
        handleToggle($input);
        break;
    default:
        jsonResponse(['success' => false, 'error' => 'Invalid action'], 400);
}

/**
 * Get all extension metadata for user
 */
function handleGet() {
    $userId = sanitize($_GET['user_id'] ?? '');
    
    if (empty($userId)) {
        jsonResponse(['success' => false, 'error' => 'Missing user_id'], 400);
    }
    
    $pdo = getConnection();
    
    try {
        $stmt = $pdo->prepare("
            SELECT id, name, version, is_enabled, installed_at, updated_at
            FROM extensions 
            WHERE user_id = ?
            ORDER BY installed_at DESC
        ");
        $stmt->execute([$userId]);
        $extensions = $stmt->fetchAll();
        
        // Convert is_enabled to boolean
        foreach ($extensions as &$ext) {
            $ext['isEnabled'] = (bool) $ext['is_enabled'];
            $ext['installedAt'] = $ext['installed_at'];
            $ext['updatedAt'] = $ext['updated_at'];
            unset($ext['is_enabled'], $ext['installed_at'], $ext['updated_at']);
        }
        
        logSync($pdo, $userId, 'extensions', count($extensions), 'download', 'success');
        
        jsonResponse([
            'success' => true,
            'extensions' => $extensions
        ]);
        
    } catch (PDOException $e) {
        jsonResponse(['success' => false, 'error' => 'Failed to fetch extensions'], 500);
    }
}

/**
 * Sync extension metadata from client
 */
function handleSync($input) {
    if (!validateRequired($input, ['user_id', 'extensions'])) {
        jsonResponse(['success' => false, 'error' => 'Missing required fields'], 400);
    }
    
    $userId = sanitize($input['user_id']);
    $extensions = $input['extensions'];
    
    $pdo = getConnection();
    
    try {
        $pdo->beginTransaction();
        $syncedCount = 0;
        
        foreach ($extensions as $ext) {
            $extId = sanitize($ext['id'] ?? '');
            if (empty($extId)) continue;
            
            $stmt = $pdo->prepare("
                INSERT INTO extensions (id, user_id, name, version, is_enabled, installed_at)
                VALUES (?, ?, ?, ?, ?, ?)
                ON DUPLICATE KEY UPDATE
                    name = VALUES(name),
                    version = VALUES(version),
                    is_enabled = VALUES(is_enabled)
            ");
            
            $stmt->execute([
                $extId,
                $userId,
                sanitize($ext['name'] ?? 'Unknown'),
                $ext['version'] ?? '1.0.0',
                ($ext['isEnabled'] ?? true) ? 1 : 0,
                $ext['installedAt'] ?? date('Y-m-d H:i:s')
            ]);
            
            $syncedCount++;
        }
        
        $pdo->commit();
        
        logSync($pdo, $userId, 'extensions', $syncedCount, 'upload', 'success');
        
        jsonResponse([
            'success' => true,
            'synced' => $syncedCount
        ]);
        
    } catch (PDOException $e) {
        $pdo->rollBack();
        logSync($pdo, $userId, 'extensions', 0, 'upload', 'failed', $e->getMessage());
        jsonResponse(['success' => false, 'error' => 'Sync failed'], 500);
    }
}

/**
 * Toggle extension enabled/disabled state
 */
function handleToggle($input) {
    if (!validateRequired($input, ['user_id', 'extension_id', 'enabled'])) {
        jsonResponse(['success' => false, 'error' => 'Missing required fields'], 400);
    }
    
    $userId = sanitize($input['user_id']);
    $extensionId = sanitize($input['extension_id']);
    $enabled = $input['enabled'] ? 1 : 0;
    
    $pdo = getConnection();
    
    try {
        $stmt = $pdo->prepare("
            UPDATE extensions 
            SET is_enabled = ? 
            WHERE id = ? AND user_id = ?
        ");
        $stmt->execute([$enabled, $extensionId, $userId]);
        
        jsonResponse([
            'success' => true,
            'updated' => $stmt->rowCount() > 0
        ]);
        
    } catch (PDOException $e) {
        jsonResponse(['success' => false, 'error' => 'Toggle failed'], 500);
    }
}
