<?php
/**
 * Dino Browser - Bookmarks Sync API
 * Handles bookmark synchronization
 */

require_once 'config.php';

$input = getInput();
$action = $input['action'] ?? $_GET['action'] ?? '';

switch ($action) {
    case 'get':
        handleGet();
        break;
    case 'sync':
        handleSync($input);
        break;
    case 'delete':
        handleDelete($input);
        break;
    default:
        jsonResponse(['success' => false, 'error' => 'Invalid action'], 400);
}

/**
 * Get all bookmarks for user
 */
function handleGet() {
    $userId = sanitize($_GET['user_id'] ?? '');
    
    if (empty($userId)) {
        jsonResponse(['success' => false, 'error' => 'Missing user_id'], 400);
    }
    
    $pdo = getConnection();
    
    try {
        // Get bookmarks
        $stmt = $pdo->prepare("
            SELECT id, url, title, favicon_url, folder_id, position, created_at, updated_at
            FROM bookmarks 
            WHERE user_id = ?
            ORDER BY position ASC, created_at DESC
        ");
        $stmt->execute([$userId]);
        $bookmarks = $stmt->fetchAll();
        
        // Get folders
        $stmt = $pdo->prepare("
            SELECT id, name, parent_id, icon, position, created_at
            FROM bookmark_folders 
            WHERE user_id = ?
            ORDER BY position ASC
        ");
        $stmt->execute([$userId]);
        $folders = $stmt->fetchAll();
        
        logSync($pdo, $userId, 'bookmarks', count($bookmarks), 'download', 'success');
        
        jsonResponse([
            'success' => true,
            'bookmarks' => $bookmarks,
            'folders' => $folders
        ]);
        
    } catch (PDOException $e) {
        jsonResponse(['success' => false, 'error' => 'Failed to fetch bookmarks'], 500);
    }
}

/**
 * Sync bookmarks from client
 */
function handleSync($input) {
    if (!validateRequired($input, ['user_id', 'bookmarks'])) {
        jsonResponse(['success' => false, 'error' => 'Missing required fields'], 400);
    }
    
    $userId = sanitize($input['user_id']);
    $bookmarks = $input['bookmarks'];
    $folders = $input['folders'] ?? [];
    
    $pdo = getConnection();
    
    try {
        $pdo->beginTransaction();
        $syncedCount = 0;
        
        // Sync folders first
        foreach ($folders as $folder) {
            $folderId = sanitize($folder['id'] ?? '');
            if (empty($folderId)) continue;
            
            $stmt = $pdo->prepare("
                INSERT INTO bookmark_folders (id, user_id, name, parent_id, icon, position, created_at)
                VALUES (?, ?, ?, ?, ?, ?, ?)
                ON DUPLICATE KEY UPDATE
                    name = VALUES(name),
                    parent_id = VALUES(parent_id),
                    icon = VALUES(icon),
                    position = VALUES(position)
            ");
            
            $stmt->execute([
                $folderId,
                $userId,
                sanitize($folder['name'] ?? 'Folder'),
                isset($folder['parentId']) ? sanitize($folder['parentId']) : null,
                $folder['icon'] ?? '📁',
                intval($folder['position'] ?? 0),
                $folder['createdAt'] ?? date('Y-m-d H:i:s')
            ]);
        }
        
        // Sync bookmarks
        foreach ($bookmarks as $bookmark) {
            $bookmarkId = sanitize($bookmark['id'] ?? '');
            if (empty($bookmarkId)) continue;
            
            $stmt = $pdo->prepare("
                INSERT INTO bookmarks (id, user_id, url, title, favicon_url, folder_id, position, created_at)
                VALUES (?, ?, ?, ?, ?, ?, ?, ?)
                ON DUPLICATE KEY UPDATE
                    url = VALUES(url),
                    title = VALUES(title),
                    favicon_url = VALUES(favicon_url),
                    folder_id = VALUES(folder_id),
                    position = VALUES(position)
            ");
            
            $stmt->execute([
                $bookmarkId,
                $userId,
                $bookmark['url'] ?? '',
                sanitize($bookmark['title'] ?? 'Untitled'),
                $bookmark['faviconUrl'] ?? null,
                isset($bookmark['folderId']) ? sanitize($bookmark['folderId']) : null,
                intval($bookmark['position'] ?? 0),
                $bookmark['createdAt'] ?? date('Y-m-d H:i:s')
            ]);
            
            $syncedCount++;
        }
        
        $pdo->commit();
        
        logSync($pdo, $userId, 'bookmarks', $syncedCount, 'upload', 'success');
        
        jsonResponse([
            'success' => true,
            'synced' => $syncedCount
        ]);
        
    } catch (PDOException $e) {
        $pdo->rollBack();
        logSync($pdo, $userId, 'bookmarks', 0, 'upload', 'failed', $e->getMessage());
        jsonResponse(['success' => false, 'error' => 'Sync failed'], 500);
    }
}

/**
 * Delete a bookmark
 */
function handleDelete($input) {
    if (!validateRequired($input, ['user_id', 'bookmark_id'])) {
        jsonResponse(['success' => false, 'error' => 'Missing required fields'], 400);
    }
    
    $userId = sanitize($input['user_id']);
    $bookmarkId = sanitize($input['bookmark_id']);
    
    $pdo = getConnection();
    
    try {
        $stmt = $pdo->prepare("DELETE FROM bookmarks WHERE id = ? AND user_id = ?");
        $stmt->execute([$bookmarkId, $userId]);
        
        jsonResponse([
            'success' => true,
            'deleted' => $stmt->rowCount() > 0
        ]);
        
    } catch (PDOException $e) {
        jsonResponse(['success' => false, 'error' => 'Delete failed'], 500);
    }
}
